#pragma once

#include <ipp.h>
#include <vector>

template<typename T> struct IppImage 
{
	T * image;

	int step;

	BOOL enable;

	IppImage() : image(NULL), step(0), enable(false) {}

	IppImage(const IppImage & i) : image(i.image), step(i.step), enable(i.enable) {}

	IppImage(T * image, int step, BOOL enable) : image(image), step(step), enable(enable) {}

	T * ptr(int x, int y) const 
	{
		return (T *)((Ipp8u *)image + y * step) + x;
	}

	T * ptr(int x, int y, int c) const
	{
		return (T *)((Ipp8u *)image + y * step) + x * c;
	}

	void safeFree()
	{
		if( image )
		{
			ippiFree( image );

			image = NULL;
		}
		step = 0;

		enable = false;
	}
};

typedef IppImage<Ipp8u> IppImage8u;
typedef IppImage<Ipp16u> IppImage16u;
typedef IppImage<Ipp32u> IppImage32u;

typedef IppImage<Ipp8s> IppImage8s;
typedef IppImage<Ipp16s> IppImage16s;
typedef IppImage<Ipp32s> IppImage32s;

typedef IppImage<Ipp32f> IppImage32f;
typedef IppImage<Ipp64f> IppImage64f;

template<typename T> struct IppRows
{
	Ipp8u* row;
	int step;

	explicit IppRows(const IppImage<T>& i) : row((Ipp8u*)i.image), step(i.step)
	{
	}

	explicit IppRows(const IppImage<T>& i, int x, int y, int c = 1) : row((Ipp8u*)i.ptr(x, y, c)), step(i.step)
	{
	}

	explicit IppRows(T* image, int step) : row((Ipp8u*)image), step(step)
	{
	}

	void operator ++()
	{
		row += step;
	}

	void operator ++(int)
	{
		row += step;
	}

	void operator +=(int r)
	{
		row += step*r;
	}
};

template<typename T, int c = 1> struct IppScanline
{
	T* ptr;

	explicit IppScanline(const IppRows<T>& i) : ptr((T*)i.row)
	{
	}

	void operator ++()
	{
		ptr += c;
	}

	void operator ++(int)
	{
		ptr += c;
	}

	void operator +=(int r)
	{
		ptr += c*r;
	}
};

template<typename T, int c = 1> struct IppScanline2
{
	T* ptr0;
	T* ptr1;

	explicit IppScanline2(const IppRows<T>& i) 
		: ptr0((T*)i.row)
		, ptr1((T*)(i.row+i.step))
	{
		
	}

	void operator ++()
	{
		ptr0 += c;
		ptr1 += c;
	}

	void operator ++(int)
	{
		ptr0 += c;
		ptr1 += c;
	}

	void operator +=(int r)
	{
		r *= c;
		ptr0 += r;
		ptr1 += r;
	}
};

template<typename T, int c = 1> struct IppScanline3
{
	T* ptr0;
	T* ptr1;
	T* ptr2;

	explicit IppScanline3(const IppRows<T>& i) 
		: ptr0((T*)(i.row-i.step))
		, ptr1((T*)i.row)
		, ptr2((T*)(i.row+i.step))
	{
		
	}

	void operator ++()
	{
		ptr0 += c;
		ptr1 += c;
		ptr2 += c;
	}

	void operator ++(int)
	{
		ptr0 += c;
		ptr1 += c;
		ptr2 += c;
	}

	void operator +=(int r)
	{
		r *= c;
		ptr0 += r;
		ptr1 += r;
		ptr2 += r;
	}
};

template<typename T, int c = 1> struct IppScanline5
{
	T* ptr0;
	T* ptr1;
	T* ptr2;
	T* ptr3;
	T* ptr4;

	IppScanline5(const IppRows<T>& i) 
		: ptr0((T*)(i.row-i.step*2))
		, ptr1((T*)(i.row-i.step))
		, ptr2((T*)i.row)
		, ptr3((T*)(i.row+i.step))
		, ptr4((T*)(i.row+i.step*2))
	{
		
	}

	void operator ++()
	{
		ptr0 += c;
		ptr1 += c;
		ptr2 += c;
		ptr3 += c;
		ptr4 += c;
	}

	void operator ++(int)
	{
		ptr0 += c;
		ptr1 += c;
		ptr2 += c;
		ptr3 += c;
		ptr4 += c;
	}

	void operator +=(int r)
	{
		r *= c;
		ptr0 += c;
		ptr1 += c;
		ptr2 += c;
		ptr3 += c;
		ptr4 += c;
	}
};

template<typename T, int c = 1> struct IppScanline4S
{
	T* ptr0;
	T* ptr1;
	T* ptr2;
	T* ptr3;

	explicit IppScanline4S(const IppRows<T>& i) 
		: ptr0((T*)(i.row))
		, ptr1((T*)(i.row+i.step))
		, ptr2((T*)(i.row+i.step*2))
		, ptr3((T*)(i.row+i.step*3))
	{
		
	}

	void operator ++()
	{
		ptr0 += c;
		ptr1 += c;
		ptr2 += c;
		ptr3 += c;
	}

	void operator ++(int)
	{
		ptr0 += c;
		ptr1 += c;
		ptr2 += c;
		ptr3 += c;
	}

	void operator +=(int r)
	{
		r *= c;
		ptr0 += r;
		ptr1 += r;
		ptr2 += r;
		ptr3 += r;
	}
};

template<typename T, int c = 1> struct IppScanline8S
{
	T* ptr0;
	T* ptr1;
	T* ptr2;
	T* ptr3;
	T* ptr4;
	T* ptr5;
	T* ptr6;
	T* ptr7;

	explicit IppScanline8S(const IppRows<T>& i) 
		: ptr0((T*)(i.row))
		, ptr1((T*)(i.row+i.step))
		, ptr2((T*)(i.row+i.step*2))
		, ptr3((T*)(i.row+i.step*3))
		, ptr4((T*)(i.row+i.step*4))
		, ptr5((T*)(i.row+i.step*5))
		, ptr6((T*)(i.row+i.step*6))
		, ptr7((T*)(i.row+i.step*7))
	{
		
	}

	void operator ++()
	{
		ptr0 += c;
		ptr1 += c;
		ptr2 += c;
		ptr3 += c;
		ptr4 += c;
		ptr5 += c;
		ptr6 += c;
		ptr7 += c;
	}

	void operator ++(int)
	{
		ptr0 += c;
		ptr1 += c;
		ptr2 += c;
		ptr3 += c;
		ptr4 += c;
		ptr5 += c;
		ptr6 += c;
		ptr7 += c;
	}

	void operator +=(int r)
	{
		r *= c;
		ptr0 += r;
		ptr1 += r;
		ptr2 += r;
		ptr3 += r;
		ptr4 += r;
		ptr5 += r;
		ptr6 += r;
		ptr7 += r;
	}
};

template<typename T, int c = 1> struct IppScanline16S
{
	T* ptr0;
	T* ptr1;
	T* ptr2;
	T* ptr3;
	T* ptr4;
	T* ptr5;
	T* ptr6;
	T* ptr7;
	T* ptr8;
	T* ptr9;
	T* ptr10;
	T* ptr11;
	T* ptr12;
	T* ptr13;
	T* ptr14;
	T* ptr15;

	explicit IppScanline16S(const IppRows<T>& i) 
		: ptr0((T*)(i.row))
		, ptr1((T*)(i.row+i.step))
		, ptr2((T*)(i.row+i.step*2))
		, ptr3((T*)(i.row+i.step*3))
		, ptr4((T*)(i.row+i.step*4))
		, ptr5((T*)(i.row+i.step*5))
		, ptr6((T*)(i.row+i.step*6))
		, ptr7((T*)(i.row+i.step*7))
		, ptr8((T*)(i.row+i.step*8))
		, ptr9((T*)(i.row+i.step*9))
		, ptr10((T*)(i.row+i.step*10))
		, ptr11((T*)(i.row+i.step*11))
		, ptr12((T*)(i.row+i.step*12))
		, ptr13((T*)(i.row+i.step*13))
		, ptr14((T*)(i.row+i.step*14))
		, ptr15((T*)(i.row+i.step*15))
	{
		
	}

	void operator ++()
	{
		ptr0 += c;
		ptr1 += c;
		ptr2 += c;
		ptr3 += c;
		ptr4 += c;
		ptr5 += c;
		ptr6 += c;
		ptr7 += c;
		ptr8 += c;
		ptr9 += c;
		ptr10 += c;
		ptr11 += c;
		ptr12 += c;
		ptr13 += c;
		ptr14 += c;
		ptr15 += c;
	}

	void operator ++(int)
	{
		ptr0 += c;
		ptr1 += c;
		ptr2 += c;
		ptr3 += c;
		ptr4 += c;
		ptr5 += c;
		ptr6 += c;
		ptr7 += c;
		ptr8 += c;
		ptr9 += c;
		ptr10 += c;
		ptr11 += c;
		ptr12 += c;
		ptr13 += c;
		ptr14 += c;
		ptr15 += c;
	}

	void operator +=(int r)
	{
		r *= c;
		ptr0 += r;
		ptr1 += r;
		ptr2 += r;
		ptr3 += r;
		ptr4 += r;
		ptr5 += r;
		ptr6 += r;
		ptr7 += r;
		ptr8 += r;
		ptr9 += r;
		ptr10 += r;
		ptr11 += r;
		ptr12 += r;
		ptr13 += r;
		ptr14 += r;
		ptr15 += r;
	}
};

template<class T> struct IppCyclicImageBuffer : public std::vector<IppImage<T> >
{
public:
	IppCyclicImageBuffer() : std::vector<IppImage<T> >(), mHead(0)
	{
	}

	IppCyclicImageBuffer(int sz) : std::vector<IppImage<T> >(sz), mHead(0)
	{
	}

protected:
	size_type mHead;

public:
	void safeFree()
	{
		for(iterator i = begin();i != end();++i)
			(*i).safeFree();
		clear();
	}

	size_type headPos() const
	{
		return mHead;
	}

	IppImage<T>& head()
	{
		return (*this)[mHead];
	}

	const IppImage<T>& head() const
	{
		return (*this)[mHead];
	}

	IppImage<T>& tail()
	{
		return (*this)[(mHead+size()-1) % size()];
	}

	const IppImage<T>& tail() const
	{
		return (*this)[(mHead+size()-1) % size()];
	}

	void rotate()
	{
		mHead = (mHead+1) % size();
	}
};

template<class T> T ippBound(T v, T a, T b)
{
	return v < a ? a : (v > b ? b : v);
}